
from detectors.evidences import ProjectDirectoriesPresent
from detectors.fileTools import _getAllDirectories, _getAllFiles
from detectors.detectorContext import DirectoryDetectorContext
from detectors.detector import *
from detectors.detector import _getAllMatchesOfEvidenceList

from codeableModels import *
from metamodels.microserviceComponentsMetamodel import *

import os

class LinkKeywordsContext(object):
    def __init__(self, linkArgs):
        self.roleName = linkArgs.pop("roleName", None)
        self.association = linkArgs.pop("association", None)
        self.stereotypeInstances = linkArgs.pop("stereotypeInstances", None)
        self.label = linkArgs.pop("label", None)
        self.taggedValues = linkArgs.pop("taggedValues", None)
        if len(linkArgs) != 0:
            raise Exception(f"unknown keywords argument")

def _guessNameFromDirectory(dir, rootDirectory):
    fileRelativePath = os.path.relpath(dir, rootDirectory)
    print("file relative path = " + fileRelativePath)
    fileNameParts = str(fileRelativePath).split("/")
    print("filenameParts = " + str(fileNameParts))
    if len(fileNameParts) >= 1:
        return fileNameParts[0].capitalize()
    return None

class DetectedComponent(object):
    def __init__(self,  dir):
        self.dir = dir
        self.evidences = []
        self.identifier = None
        self.name = None
        self.aliases = []
        self.componentTypes = []
        self.technologyTypes = []
        self.linkTypes = []

    # if a link is implied by the component detected, this method returns the corresponding
    # link evidence
    def getImpliedLinkEvidence(self, source, target):
        return LinkEvidence(source, target, _getAllMatchesOfEvidenceList(self.evidences), 
                linkTypes = self.linkTypes, technologyTypes = self.technologyTypes)

    def addTypes(self, evidence):
        for t in evidence.componentTypes:
            if t not in self.componentTypes:
                self.componentTypes.append(t)
        for t in evidence.technologyTypes:
            if t not in self.technologyTypes:
                self.technologyTypes.append(t)
        for t in evidence.linkTypes:
            if t not in self.linkTypes:
                self.linkTypes.append(t)

    def computeIdentifier(self):
        name = self.name
        if name == None or name == "":
            return ""
        # remove all white space and lower cases first letter
        self.identifier = ''.join(name.split())
        # lower the word only if it is not a capitalized word, like REST
        if not self.identifier[1].isupper():
            self.identifier = self.identifier[0].lower() + self.identifier[1:]

    def addNameAndComputeIdAndAliases(self, name, project):
        self.name = name
        self.computeIdentifier()
        self.aliases = [name, self.identifier]
        for envVar in project.envVars:
            key = envVar.lower()
            value = project.envVars[envVar].lower()
            if (self.identifier.lower() in key or   
                self.name.lower() in key or
                self.identifier.lower() in value or   
                self.name.lower() in value):
                self.aliases.append(key)
                self.aliases.append(value) 

class Project(object):
    def __init__(self, name, rootDirectory):
        self.name = name
        self.rootDirectory = rootDirectory
        self.genModel = ""
        self.allDirectories = _getAllDirectories(rootDirectory)
        self.allDirectories.append(os.path.realpath(rootDirectory))
        self.allFiles = _getAllFiles(rootDirectory)
        self.allPaths = self.allDirectories + self.allFiles
        self.failedEvidences = []
        self.objectNames = []
        self.detectedComponents = {}
        self.envVars = {}

    def checkRequiredEvidences(self, evidenceIdentifier, projectDirectories, requiredEvidences):
        failedEvidences = []

        modelElementDirectories = None
        if projectDirectories != None:
            projectDirectoriesPresentEvidence = ProjectDirectoriesPresent(projectDirectories)
            failedEvidences.extend(projectDirectoriesPresentEvidence.check(self, evidenceIdentifier))
            modelElementDirectories = projectDirectoriesPresentEvidence.modelElementDirectories

        if modelElementDirectories == None:
            modelElementDirectories = []
   
        if requiredEvidences != None:
            for evidence in requiredEvidences:
                print("checking evidence = " + str(evidence))
                evidence.modelElementDirectories = modelElementDirectories
                failedEvidences.extend(evidence.check(self, evidenceIdentifier))
            self.failedEvidences.extend(failedEvidences)

        return failedEvidences

    def createComponent(self, name, componentTypes, projectDirectories, requiredEvidences):
        if projectDirectories != None and not isinstance(projectDirectories, list):
            projectDirectories = [projectDirectories]
        if requiredEvidences != None and not isinstance(requiredEvidences, list):
            requiredEvidences = [requiredEvidences]
        if isinstance(componentTypes, list):
            componentTypesString = "[" + ", ".join(componentTypes) + "]"
        else:
            componentTypesString = componentTypes

        failedEvidences = self.checkRequiredEvidences(name, projectDirectories, requiredEvidences)

        if failedEvidences == []:
            # remove all white space and lower cases first letter
            identifier = ''.join(name.split())
            # lower the word only if it is not a capitalized word, like REST
            if not identifier[1].isupper():
                identifier = identifier[0].lower() + identifier[1:]
            self.genModel += f"{identifier} = CClass(component, '{name}', stereotypeInstances = {componentTypesString})\n"
            self.objectNames.append(identifier)
            return identifier
        else:
            print("FAILED EVIDENCE: " + str(failedEvidences))
            return None

    def addComponent(self, detectors, dir, **kwargs):
        options = {'name' : None}
        options.update(kwargs)
        name = options["name"]

        if not isinstance(detectors, list):
            detectors = [detectors]
        dir = os.path.realpath(self.rootDirectory + "/" + dir)
        detectedComponent = DetectedComponent(dir)

        for detector in detectors:
            try:
                evidences = detector.detect(dir)
                if len(evidences) == 1:
                    detectedComponent.addTypes(evidences[0])
                    detectedComponent.evidences.extend(evidences)
                    # use detector guessed name, if none was provided or guess by previous detector
                    if name == None and evidences[0].name != None:
                        name = evidences[0].name
            except DetectorException as de:
                self.failedEvidences.append("detector '" + detector.__class__.__name__ + "' failed in directory '" + dir + 
                    "': " + str(de))
                return None                    
            if len(evidences) == 0:
                self.failedEvidences.append("detector '" + detector.__class__.__name__ + "' failed in directory '" + dir + 
                    "': no component found")
                return None
            if len(evidences) > 1:
                self.failedEvidences.append("detector '" + detector.__class__.__name__ + "' failed in directory '" + dir + 
                    "': multiple components found")
                return None

        if name == None:
            # no dectector had a name guess => guess from directory
            name = _guessNameFromDirectory(dir, self.rootDirectory)

        componentTypesString = "[" + ", ".join(detectedComponent.componentTypes) + "]"
        detectedComponent.addNameAndComputeIdAndAliases(name, self)
        identifier = detectedComponent.identifier
        self.genModel += f"{identifier} = CClass(component, '{name}', stereotypeInstances = {componentTypesString})\n"
        self.objectNames.append(identifier)
        
        self.detectedComponents[identifier] = detectedComponent
        return identifier
        
    def addLink(self, detectors, source, target):
        print("*** ADD LINK: " + str(source) + " -> " + str(target))
        if not isinstance(detectors, list):
            detectors = [detectors]
        if source == None:
            raise DetectorException("addLink: source cannot be 'None'")
        if target == None:
            raise DetectorException("addLink: target cannot be 'None'")
        # if not isinstance(targets, list):
        #     targets = [targets]
        # targetsString = "[" + ", ".join(targets) + "]"
        dir = self.detectedComponents[source].dir
        evidences = []

        for detector in detectors:
            if isinstance(detector, str):
                if detector in self.detectedComponents:
                    # a component identifier can be provided as a detector, if the compoent's matches might
                    # be considered as enough evidence for a link as well (e.g., for a web server
                    # offering the component already implies possible links to the web clients)
                    evidences.append(self.detectedComponents[detector].getImpliedLinkEvidence(source, target))
                else:
                    self.failedEvidences.append("detector '" + detector.__class__.__name__ + "' for link '" + source + "' to '" + target + 
                        ": component '" + detector + "' not yet detected")
            else:            
                try:
                    evidences.extend(detector.detect(dir, source = self.detectedComponents[source], 
                        target = self.detectedComponents[target]))
                except DetectorException as de:
                    self.failedEvidences.append("detector '" + detector.__class__.__name__ + "' for link '" + source + "' to '" + target + 
                        "' failed in directory '" + dir + "': " + str(de))

        if len(evidences) > 0:
            # if there are multiple link evidences, they are all created by the same detector:
            # just use the first to get the stereotypes etc.
            linkEvidence = evidences[0]
            if not isinstance(linkEvidence, LinkEvidence):
                self.failedEvidences.append("evidence '" + str(linkEvidence) + "' should be a link evidence, but it is not")
                return

            keywordsArgs = "roleName = \"target\""
            if len(linkEvidence.linkTypes) > 0:
                keywordsArgs = keywordsArgs + ", stereotypeInstances = [" + ", ".join(linkEvidence.linkTypes) + "]"

        # if failedEvidences == []:
        #     linkKeywords = LinkKeywordsContext(linkArgs)
        #     keywordsArgsList = []
        #     if linkKeywords.roleName != None:
        #         keywordsArgsList.append(f"roleName = \"{linkKeywords.roleName}\"")
        #     if linkKeywords.association != None:
        #         keywordsArgsList.append(f"association = {linkKeywords.association}")
        #     if linkKeywords.stereotypeInstances != None:
        #         keywordsArgsList.append(f"stereotypeInstances = [{linkKeywords.stereotypeInstances}]")
        #     if linkKeywords.label != None:
        #         keywordsArgsList.append(f"label = \"{linkKeywords.label}\"")
        #     if linkKeywords.taggedValues != None:
        #         keywordsArgsList.append(f"taggedValues = {{{linkKeywords.taggedValues}}}")
        #     if len(keywordsArgsList) > 0:
        #         keywordsArgsString = ", " + ", ".join(keywordsArgsList)

            self.genModel += f"addLinks({{{source}: {target}}}, {keywordsArgs})\n"        

    def addLinks(self, detectors, source, targets):
        for target in targets:
            self.addLink(detectors, source, target)


    def createLinks(self, source, targets, linkArgs, projectDirectories, requiredEvidences):
        print("*** ADD LINKS: " + str(source) + " -> " + str(targets))
        if targets == None:
            self.failedEvidences.append("cannot check for links from '" + str(source) + "' as 'targets' is 'None'")
            print("FAILED EVIDENCE: " + str(self.failedEvidences))
            return  

        if not isinstance(targets, list):
            targets = [targets]

        if projectDirectories != None and not isinstance(projectDirectories, list):
            projectDirectories = [projectDirectories]
        if requiredEvidences != None and not isinstance(requiredEvidences, list):
            requiredEvidences = [requiredEvidences]
        targetsString = "[" + ", ".join(targets) + "]"

        failedEvidences = self.checkRequiredEvidences(f"link '{source}->{targetsString}'", projectDirectories, requiredEvidences)

        if failedEvidences == []:
            linkKeywords = LinkKeywordsContext(linkArgs)
            keywordsArgsList = []
            if linkKeywords.roleName != None:
                keywordsArgsList.append(f"roleName = \"{linkKeywords.roleName}\"")
            if linkKeywords.association != None:
                keywordsArgsList.append(f"association = {linkKeywords.association}")
            if linkKeywords.stereotypeInstances != None:
                keywordsArgsList.append(f"stereotypeInstances = [{linkKeywords.stereotypeInstances}]")
            if linkKeywords.label != None:
                keywordsArgsList.append(f"label = \"{linkKeywords.label}\"")
            if linkKeywords.taggedValues != None:
                keywordsArgsList.append(f"taggedValues = {{{linkKeywords.taggedValues}}}")
            if len(keywordsArgsList) > 0:
                keywordsArgsString = ", " + ", ".join(keywordsArgsList)

            self.genModel += f"addLinks({{{source}: {targetsString}}}{keywordsArgsString})\n"
        else:
            print("FAILED EVIDENCE: " + str(failedEvidences))

    def saveAsFile(self, fileName):
        source = "from codeableModels import *\n"
        source += "from metamodels.microserviceComponentsMetamodel import *\n\n"
        source += self.genModel
        objects = "[" + ", ".join(self.objectNames) + "]"
        source += f"\n\nmodel = CBundle(\"model\", elements = {objects})"
        source += "\nmodelViews = [model, {}]\n"
        print("\n\nWriting File: \n\n" +source)
        f = open(fileName, "w")
        f.write(source)
        f.close()
        
    def detectServices(self):
        detector = Detector(self)
        detector.detectServicesInFiles()
        print("FOUND SERVICES: " + str(detector.detectedServices))
        #self.serviceShortNames = {}
        #for service in detector.detectedServices:
        #    self.serviceShortNames[service] = self.createComponent(service.name, "service", None, None)
        return detector

    def addEnvVars(self, detectors, dir):
        if not isinstance(detectors, list):
            detectors = [detectors]
        evidences = []

        dir = os.path.realpath(self.rootDirectory + "/" + dir)

        for detector in detectors:
            try:
                evidences.extend(detector.detect(dir))
            except DetectorException as de:
                self.failedEvidences.append("detector '" + detector.__class__.__name__ + "' for directory + '" + 
                    dir + "' failed: " + str(de))

        for evidence in evidences:
            self.envVars.update(evidence.map)

    def addComponentToModel(self, evidence):
        stereotypes = "[" + ", ".join(evidence.componentTypes) + "]"
        # remove all white space and lower cases first letter
        identifier = ''.join(evidence.name.split())
        # lowercase the word only if it is not a capitalized word, like REST
        if not identifier[1].isupper():
            identifier = identifier[0].lower() + identifier[1:]
        self.genModel += f"{identifier} = CClass(component, '{evidence.name}', stereotypeInstances = {stereotypes})\n"
        self.objectNames.append(identifier)
        self.modelIdentifiers[evidence] = identifier

    def createModel(self, evidences):
        self.genModel = ""
        self.modelIdentifiers = {}
        for evidence in evidences:
            if isinstance(evidence, ComponentEvidence):
                self.addComponentToModel(evidence)


    def detectAndGenerateServiceModel(self):
        detector = self.detectServices()
        self.createModel(detector.detectedServices)
